/*
 * Decompiled with CFR 0.152.
 */
package org.genericsystem.kernel;

import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.NavigableSet;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import org.genericsystem.kernel.Generic;
import org.genericsystem.kernel.GenericHandler;
import org.genericsystem.kernel.Root;
import org.genericsystem.kernel.Transaction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Archiver {
    public static final Logger log = LoggerFactory.getLogger(Archiver.class);
    private static final long ARCHIVER_COEFF = 5L;
    private static final String PATTERN = "yyyy.MM.dd_HH-mm-ss.SSS";
    private static final String MATCHING_REGEX = "[0-9]{4}.[0-9]{2}.[0-9]{2}_[0-9]{2}-[0-9]{2}-[0-9]{2}.[0-9]{3}---[0-9]+";
    protected static final String GS_EXTENSION = ".gs";
    protected static final String ZIP_EXTENSION = ".zip";
    private static final String PART_EXTENSION = ".part";
    private static final String LOCK_FILE_NAME = ".lock";
    private static final long SNAPSHOTS_PERIOD = 1000L;
    private static final long SNAPSHOTS_INITIAL_DELAY = 1000L;
    private final ScheduledExecutorService scheduler = Executors.newSingleThreadScheduledExecutor();
    protected final Root root;
    private final File directory;
    private FileLock lockFile;
    private final ZipFileManager zipFileManager = new ZipFileManager(new FileManager());

    public static String getFileExtension() {
        return ".gs.zip";
    }

    public Archiver(Root root, String directoryPath) {
        String snapshotPath;
        this.root = root;
        this.directory = this.prepareAndLockDirectory(directoryPath);
        if (this.directory != null && (snapshotPath = this.getSnapshotPath(this.directory)) != null) {
            try {
                this.getLoader(this.zipFileManager.getObjectInputStream(snapshotPath + Archiver.getFileExtension())).loadSnapshot();
            }
            catch (IOException | ClassNotFoundException e) {
                log.error(e.getMessage(), (Throwable)e);
            }
        }
        this.startScheduler();
    }

    protected Loader getLoader(ObjectInputStream objectInputStream) {
        return new Loader(objectInputStream);
    }

    protected Saver getSaver(ObjectOutputStream objectOutputStream, long ts) {
        return new Saver(objectOutputStream, ts);
    }

    private Archiver startScheduler() {
        if (this.directory != null && this.lockFile != null) {
            this.scheduler.scheduleAtFixedRate(new Runnable(){

                @Override
                public void run() {
                    try {
                        Archiver.this.doSnapshot();
                    }
                    catch (IOException e) {
                        log.error(e.getMessage(), (Throwable)e);
                    }
                }
            }, 1000L, 1000L, TimeUnit.MILLISECONDS);
        }
        return this;
    }

    public void close() {
        if (this.directory != null && this.lockFile != null) {
            this.scheduler.shutdown();
            try {
                this.doSnapshot();
                this.lockFile.close();
                this.lockFile = null;
            }
            catch (IOException e) {
                throw new IllegalStateException(e);
            }
        }
    }

    private void doSnapshot() throws IOException {
        long ts = this.root.pickNewTs();
        String fileName = this.directory.getAbsolutePath() + File.separator + Archiver.getFilename(ts) + Archiver.getFileExtension();
        String partFileName = fileName + PART_EXTENSION;
        this.getSaver(this.zipFileManager.getObjectOutputStream(partFileName, Archiver.getFilename(ts) + GS_EXTENSION), ts).saveSnapshot();
        new File(partFileName).renameTo(new File(fileName));
        this.manageOldSnapshots(this.directory);
    }

    private void manageOldSnapshots(File directory) {
        long firstTs;
        NavigableMap<Long, File> snapshotsMap = Archiver.snapshotsMap(directory, Archiver.getFileExtension());
        long lastTs = (Long)snapshotsMap.lastKey();
        long ts = firstTs = ((Long)snapshotsMap.firstKey()).longValue();
        Iterator iterator = new TreeSet(snapshotsMap.keySet()).iterator();
        while (iterator.hasNext()) {
            long snapshotTs = (Long)iterator.next();
            if (snapshotTs == lastTs || snapshotTs == firstTs) continue;
            if (snapshotTs - ts < this.minInterval(lastTs - snapshotTs)) {
                this.removeSnapshot(snapshotsMap, snapshotTs);
                continue;
            }
            ts = snapshotTs;
        }
    }

    private long minInterval(long periodNumber) {
        return (long)Math.floor(periodNumber / 5L);
    }

    private void removeSnapshot(NavigableMap<Long, File> snapshotsMap, long ts) {
        ((File)snapshotsMap.get(ts)).delete();
        snapshotsMap.remove(ts);
    }

    private File prepareAndLockDirectory(String directoryPath) {
        if (directoryPath == null) {
            return null;
        }
        File directory = new File(directoryPath);
        if (directory.exists()) {
            if (!directory.isDirectory()) {
                throw new IllegalStateException("Datasource path : " + directoryPath + " is not a directory");
            }
        } else if (!directory.mkdirs()) {
            throw new IllegalStateException("Can't make directory : " + directoryPath);
        }
        try {
            this.lockFile = new FileOutputStream(directoryPath + File.separator + LOCK_FILE_NAME).getChannel().tryLock();
            return directory;
        }
        catch (IOException | OverlappingFileLockException e) {
            throw new IllegalStateException("Locked directory : " + directoryPath);
        }
    }

    private String getSnapshotPath(File directory) {
        NavigableMap<Long, File> snapshotsMap = Archiver.snapshotsMap(directory, Archiver.getFileExtension());
        return snapshotsMap.isEmpty() ? null : directory.getAbsolutePath() + File.separator + Archiver.getFilename((Long)snapshotsMap.lastKey());
    }

    private static NavigableMap<Long, File> snapshotsMap(File directory, String extension) {
        TreeMap<Long, File> snapshotsMap = new TreeMap<Long, File>();
        for (File file : directory.listFiles()) {
            String filename = file.getName();
            if (file.isDirectory() || !filename.endsWith(extension) || !(filename = filename.substring(0, filename.length() - extension.length())).matches(MATCHING_REGEX)) continue;
            try {
                snapshotsMap.put(Archiver.getTimestamp(filename), file);
            }
            catch (ParseException pe) {
                throw new IllegalStateException(pe);
            }
        }
        return snapshotsMap;
    }

    private static long getTimestamp(String filename) throws ParseException {
        return Long.parseLong(filename.substring(filename.lastIndexOf("---") + 3));
    }

    private static String getFilename(long ts) {
        return new SimpleDateFormat(PATTERN).format(new Date(ts / 1000000L)) + "---" + ts;
    }

    protected static class FileManager {
        protected FileManager() {
        }

        protected FileOutputStream getFileOutputStream(String fileName) throws IOException {
            return new FileOutputStream(fileName);
        }

        protected FileInputStream getFileInputStream(String fileName) throws IOException {
            return new FileInputStream(new File(fileName));
        }
    }

    protected static class ZipFileManager {
        private final FileManager fileManager;

        protected ZipFileManager(FileManager fileManager) {
            this.fileManager = fileManager;
        }

        protected ObjectOutputStream getObjectOutputStream(String zipFileName, String fileName) throws IOException {
            ZipOutputStream zipOutput = new ZipOutputStream(this.fileManager.getFileOutputStream(zipFileName));
            zipOutput.putNextEntry(new ZipEntry(fileName));
            return new ObjectOutputStream(zipOutput);
        }

        protected ObjectInputStream getObjectInputStream(String fileName) throws IOException {
            ZipInputStream inputStream = new ZipInputStream(this.fileManager.getFileInputStream(fileName));
            inputStream.getNextEntry();
            return new ObjectInputStream(inputStream);
        }
    }

    protected class Loader {
        protected final ObjectInputStream objectInputStream;
        protected final Transaction transaction;

        protected Loader(ObjectInputStream objectInputStream) {
            this.objectInputStream = objectInputStream;
            this.transaction = (Transaction)Archiver.this.root.buildTransaction();
        }

        public Transaction getTransaction() {
            return this.transaction;
        }

        private void loadSnapshot() throws ClassNotFoundException, IOException {
            try {
                HashMap<Long, Generic> vertexMap = new HashMap<Long, Generic>();
                while (true) {
                    this.loadDependency(vertexMap);
                }
            }
            catch (EOFException eOFException) {
                return;
            }
        }

        protected long loadTs() throws IOException {
            return this.objectInputStream.readLong();
        }

        protected long[] loadOtherTs() throws IOException {
            return new long[]{this.objectInputStream.readLong(), this.objectInputStream.readLong(), this.objectInputStream.readLong()};
        }

        protected void loadDependency(Map<Long, Generic> vertexMap) throws IOException, ClassNotFoundException {
            long ts = this.loadTs();
            long[] otherTs = this.loadOtherTs();
            if (otherTs[0] == 0L) {
                otherTs[0] = 1L;
            }
            Serializable value = (Serializable)this.objectInputStream.readObject();
            Generic meta = this.loadAncestor(ts, vertexMap);
            List<Generic> supers = this.loadAncestors(ts, vertexMap);
            List<Generic> components = this.loadAncestors(ts, vertexMap);
            vertexMap.put(ts, (Generic)new GenericHandler.SetArchiverHandler<Generic>(ts, this.transaction, meta, supers, value, components, otherTs).resolve());
            assert (this.getTransaction().isAlive(vertexMap.get(ts))) : vertexMap.get(ts).info();
        }

        protected List<Generic> loadAncestors(long ts, Map<Long, Generic> vertexMap) throws IOException {
            ArrayList<Generic> ancestors = new ArrayList<Generic>();
            int sizeComponents = this.objectInputStream.readInt();
            for (int j = 0; j < sizeComponents; ++j) {
                ancestors.add(this.loadAncestor(ts, vertexMap));
            }
            return ancestors;
        }

        protected Generic loadAncestor(long ts, Map<Long, Generic> vertexMap) throws IOException {
            long designTs = this.objectInputStream.readLong();
            Generic ancestor = vertexMap.get(designTs);
            assert (ancestor != null || designTs == ts);
            return ancestor;
        }
    }

    public class Saver {
        protected final ObjectOutputStream objectOutputStream;
        protected final Transaction transaction;

        protected Saver(ObjectOutputStream objectOutputStream, long ts) {
            this.objectOutputStream = objectOutputStream;
            this.transaction = this.buildTransaction(ts);
        }

        protected Transaction buildTransaction(long ts) {
            return new Transaction(Archiver.this.root, ts);
        }

        public Transaction getTransaction() {
            return this.transaction;
        }

        private void saveSnapshot() throws IOException {
            this.writeDependencies(this.transaction.computeDependencies(Archiver.this.root), new HashSet<Generic>());
            this.objectOutputStream.flush();
            this.objectOutputStream.close();
        }

        private void writeDependencies(NavigableSet<Generic> dependencies, Set<Generic> vertexSet) throws IOException {
            for (Generic dependency : dependencies) {
                if (!vertexSet.add(dependency)) continue;
                this.writeDependency(dependency);
            }
        }

        private void writeDependency(Generic dependency) throws IOException {
            this.writeAncestorId(dependency, dependency);
            this.writeOtherTs(dependency);
            this.objectOutputStream.writeObject(dependency.getValue());
            this.writeAncestorId(dependency, dependency.getMeta());
            this.writeAncestorsId(dependency, dependency.getSupers());
            this.writeAncestorsId(dependency, dependency.getComponents());
        }

        protected void writeOtherTs(Generic dependency) throws IOException {
            this.objectOutputStream.writeLong(dependency.getLifeManager().getBirthTs());
            this.objectOutputStream.writeLong(dependency.getLifeManager().getLastReadTs());
            this.objectOutputStream.writeLong(dependency.getLifeManager().getDeathTs());
        }

        private void writeAncestorsId(Generic dependency, List<Generic> ancestors) throws IOException {
            this.objectOutputStream.writeInt(ancestors.size());
            for (Generic ancestor : ancestors) {
                this.writeAncestorId(dependency, ancestor);
            }
        }

        protected void writeAncestorId(Generic dependency, Generic ancestor) throws IOException {
            this.objectOutputStream.writeLong(ancestor != null ? ancestor.getTs() : dependency.getTs());
        }
    }
}

